%NOISE   Add noise to an image
%
% SYNOPSIS:
%  image_out = noise(image_in, noise_type, parameter1, parameter2)
%
% PARAMETERS:
%  noise_type: 'gaussian', 'uniform', 'poisson', 'binary', 'saltpepper'
%  The parameters vary for the differnt noise types.
%
%  Gaussian:
%    parameter1 = standard deviation of the noise
%  Uniform:
%    parameter1 = lower bound
%    parameter2 = upper bound
%  Poisson:
%    parameter1 = conversion
%     The intensity of the input image divided by the conversion
%     variable is used as mean value for the Poisson distribution.
%     The conversion factor can be used to relate the pixel values
%     with the number of counts. For example, to simulate a photon
%     limited image acquired by a CCD camera, the conversion factor
%     specifies the relation between the number of photons recorded
%     and the pixel value it is represented by.
%  Binary:
%    parameter1 = probability for a 1->0 transition
%    parameter2 = probability for a 0->1 transition
%     Image_in must be binary.
%  Salt & pepper:
%    parameter1 = probability for a pixel to become 255
%    parameter2 = probability for a pixel to become 0
%
% DEFAULTS:
%  noise_type = 'gaussian'
%  parameter1  = 1
%  parameter2  = 0

% (C) Copyright 1999-2008               Pattern Recognition Group
%     All rights reserved               Faculty of Applied Physics
%                                       Delft University of Technology
%                                       Lorentzweg 1
%                                       2628 CJ Delft
%                                       The Netherlands
%
% Bernd Rieger, March 2001.
% 16 October 2008: Added 'saltpepper' (CL).

function out = noise(varargin)

d = struct('menu','Generation',...
           'display','Add noise',...
           'inparams',struct('name',       {'in',         'method',    'par1',      'par2'},...
                             'description',{'Input image','Noise Type','Parameter 1','Parameter 2'},...
                             'type',       {'image',      'option',    'array',     'array'},...
                             'dim_check',  {0,            0,           0,           0},...
                             'range_check',{[],{'gaussian','uniform','poisson','binary','saltpepper'},'R','R'},...
                             'required',   {1,            0,           0,           0},...
                             'default',    {'a',          'gaussian',  1,           0}...
                            ),...
           'outparams',struct('name',{'out'},...
                              'description',{'Output image'},...
                              'type',{'image'}...
                              )...
          );
if nargin == 1
   s = varargin{1};
   if ischar(s) & strcmp(s,'DIP_GetParamList')
      out = d;
      return
   end
end
try
   [in,method,par1,par2] = getparams(d,varargin{:});
catch
   if ~isempty(paramerror)
      error(paramerror)
   else
      error(firsterr)
   end
end

switch method
   case 'gaussian'
      out = dip_gaussiannoise(in,par1*par1); % for unknown reason DIPlib chose here variance
   case 'uniform'
      out = dip_uniformnoise(in, par1, par2);
   case 'poisson'
      if  par1<=0
         error('Conversion greater than zero.');
      end
      out = dip_poissonnoise(in, par1);
   case 'binary'
      if par1>1 | par2 >1 | par1<0 | par2<0
         error('Transition probability between 0 and 1.');
      end
      out = dip_binarynoise(in, par1, par2);
   case 'saltpepper'
      n = dip_uniformnoise(newim(in), 0, 1);
      out = in;
      out(n>(1-par1)) = 255;
      out(n<par2) = 0;
   otherwise
      error('Unkown noise type.')
end
