%RESAMPLE   Resample an image
%
% SYNOPSIS:
%  image_out = resample(image_in, zoom, translation, interpolation_method)
%
% PARAMETERS:
%  zoom        = array containing a zoom
%  translation = array containing a shift
%  interpolation_method = 'bspline', '4-cubic', '3-cubic', 'linear', 'zoh'
%
% MATHEMATICAL OPERATION:
%  With zoom and translation as vectors, and pos the position vector:
%  image_out(pos)=image_in( (pos+translation)/zoom );
%
%  This one of few DIPimage functions that will not immediately convert
%  its input to a floating point type for better accuracy in the result. The
%  philosophy is that it may be used by display related functions to scale
%  uint8 display images for example.
%
%  To prevent clipping artifacts you are strongly recommended to convert
%  your input data to a floating point type explicitly, e.g.
%  image_in=dip_image(image_in,'sfloat');
%
% DEFAULTS:
%  zoom        = 2
%  translation = 0
%  interpolation_method = 'linear'
%
% SEE ALSO:
%  shift

% (C) Copyright 1999-2007               Pattern Recognition Group
%     All rights reserved               Faculty of Applied Physics
%                                       Delft University of Technology
%                                       Lorentzweg 1
%                                       2628 CJ Delft
%                                       The Netherlands
%
% Bernd Rieger, March 2001.
% ?? Changed default interpolation to linear
% 14 March 2005: Fixed zoom==0 problem.
% Nov 2007 - MvG - Added MATHEMATICAL OPERATION (including warning) section
%                  to the help.
% Jan 2009 Bug fix in the computation of the pixel size

function out = resample(varargin)

d = struct('menu','Manipulation',...
           'display','Resample',...
           'inparams',struct('name',       {'in','zoom','trans','interp'},...
                             'description',{'Input image','Zoom','Translation','Interpolation method'},...
                             'type',       {'image','array','array','option'},...
                             'dim_check',  {0,1,1,0},...
                             'range_check',{[],[realmin,+Inf],'R',{'linear','4-cubic','3-cubic','bspline','zoh'}},...
                             'required',   {1,0,0,0},...
                             'default',    {'a',2,0,'linear'}...
                            ),...
           'outparams',struct('name',{'out'},...
                              'description',{'Output image'},...
                              'type',{'image'}...
                              )...
          );
if nargin == 1
   s = varargin{1};
   if ischar(s) & strcmp(s,'DIP_GetParamList')
      out = d;
      return
   end
end
if nargin >= 4 & isequal(varargin{4},'bilinear')
   varargin{4} = 'linear';
end
try
   [in,zoom,trans,interp] = getparams(d,varargin{:});
catch
   if ~isempty(paramerror)
      error(paramerror)
   else
      error(firsterr)
   end
end
out = iterate('dip_resampling',in,zoom,trans,interp);
out.pixelsize = in.pixelsize./zoom;
