%ROTATION   Rotate a 2D/3D image around an axis
%
% The rotation center is in the center of the image.
% Image rotation by means of shears
%
% SYNOPSIS:
%  image_out = rotation(image_in, angle, axis, method, bgval)
%
%   axis
%      2D: meaningless
%      3D: 1 - x-axis, 2 - y-axis, 3 - z-axis
%   method
%      Interpolation method. String containing one of the following values:
%      'default', 'bspline', '4-cubic', '3-cubic', 'linear', 'zoh'.
%   bgval
%      Value used to fill up the background. String containing one of the
%      following values:
%      'zero', 'min', 'max'.
%
% DEFAULTS:
%   axis   ='3' z-axis
%   method ='bspline'
%   bgval  ='zero'
%
%  Note: White border around the rotated image can occur if image_in
%  is of type integer, convert to float
%
%  Sign of the 2D rotation: implementation in the mathmetical sense, but note the
%  y-axis is positive downwards! Thus: left turning has negative sign, and
%  right positive.
%
% SEE ALSO: rotation3d, dip_image/rotate

% (C) Copyright 1999-2001               Pattern Recognition Group
%     All rights reserved               Faculty of Applied Physics
%                                       Delft University of Technology
%                                       Lorentzweg 1
%                                       2628 CJ Delft
%                                       The Netherlands
%
% Bernd Rieger, Feb 2001

function out = rotation(varargin)

d = struct('menu','Manipulation',...
           'display','Rotation around an axis',...
           'inparams',struct('name',{'in',   'angle', 'axis',  'method',     'bgval'},...
                  'description',{'Input image','Angle (rad)', 'Axis',     'Interpolation Method',     'Background Value'},...
                  'type',       {'image',      'array',     'option',     'option',            'option'},...
                  'dim_check',  {0,            0,           0,             0,             0},...
                  'range_check',{[],           [], {1,2,3}, {'bspline', '4-cubic', '3-cubic', 'linear', 'zoh'},{'zero', 'min', 'max'}},...
                  'required',   {1,            1,           0,             0,                  0},...
                  'default',    {'a',          pi/4,        3,           'bspline',       'zero'}...
                   ),...
           'outparams',struct('name',{'out'},...
                   'description',{'Output image'},...
                   'type',{'image'}...
                   )...
           );
if nargin == 1
   s = varargin{1};
   if ischar(s) & strcmp(s,'DIP_GetParamList')
      out = d;
      return
   end
end
if nargin >= 4 & isequal(varargin{4},'bilinear')
   varargin{4} = 'linear';
end
try
   [in,angle, axis,method,bgval] = getparams(d,varargin{:});
catch
   if ~isempty(paramerror)
      error(paramerror)
   else
      error(firsterr)
   end
end
d = ndims(in{1});
if d == 2
   out =  iterate('dip_rotation',in,angle,method,bgval);
elseif d == 3
   out = dip_rotation3d_axis(in,angle,axis-1,method,bgval);
else
   error('Rotation only supported for 2D/3D images.');
end
