%MEASURE_GAMMA_MONITOR   Interactive tool to measure the gamma of your monitor
%
% SYNOPSIS:
%  gamma_RGB = measure_gamma_monitor(gamma_ramp)
%
% SEE ALSO:
%  dipsetpref('Gamma') and dipsetpref('GammaGrey')

% (C) Copyright 1999-2007               Pattern Recognition Group
%     All rights reserved               Faculty of Applied Physics
%                                       Delft University of Technology
%                                       Lorentzweg 1
%                                       2628 CJ Delft
%                                       The Netherlands
%
% Judith Dijk.
% February 2007, added dippref('Gamma') (BR)
% 22 October 2007, rewritten for a more smooth user experience. (CL)

function gamma = measure_gamma_monitor(gamma_ramp)

if nargin==1 & ischar(gamma_ramp) & strcmp(gamma_ramp,'DIP_GetParamList') % Avoid being in menu
   gamma = struct('menu','none');
   return
end

if matlabversion<6
   error('Functionality not supported on your version of MATLAB. Sorry.')
end

% Get current preference value, and limit to the slider range we're implementing.
gamma = dipgetpref('Gamma');
gamma = min(gamma,5);
gamma = max(gamma,1/5);

% Constants
imsize = [400,30];
edgesize = [20,10];
figsize = [imsize(1)+2*edgesize(1),500];

% Real ramp and noise ramp images
if nargin<1
   gamma_ramp = 2;
end
rampim = xx(imsize,'corner');
rampim = (rampim/max(rampim))^gamma_ramp;
nrampim = noise(rampim,'uniform',0,1);
nrampim = nrampim>1;
rampim = double(rampim);
nrampim = double(nrampim);

% Create figure
figpos = get(0,'screensize');
figpos = [round((figpos(3:4)-figsize)/2),figsize];
figh = figure('Units','pixels',...
              'Position',figpos,...
              'ToolBar','none',...
              'MenuBar','none',...
              'Resize','off',...
              'BusyAction','cancel',...
              'Color',get(0,'defaultuicontrolbackgroundcolor'),...
              'CloseRequestFcn','delete(gcbf)',...
              'HandleVisibility','off',...
              'IntegerHandle','off',...
              'Interruptible','off',...
              'Name','DIPimage monitor gamma measuring tool',...
              'NumberTitle','off');

% Create controls
cancel = uicontrol('Parent',figh,...
                   'Units','pixels',...
                   'Style','pushbutton',...
                   'String','Cancel',...
                   'CallBack','delete(gcbf)');

ctrsz = get(cancel,'Extent');
ctrsz = ctrsz(3:4)+[10,4];
pos = [figsize(1)-ctrsz(1)-edgesize(1),edgesize(2),ctrsz];
set(cancel,'Position',pos);
pos(1) = pos(1)-ctrsz(1)-edgesize(2);
ctrsz = ctrsz-[10,4];

submit = uicontrol('Parent',figh,...
                   'Units','pixels',...
                   'Style','pushbutton',...
                   'Position',pos,...
                   'String','OK',...
                   'CallBack','set(gcbf,''UserData'',1)');

top = pos(2)+pos(4)+edgesize(2);
panelheight = round((figsize(2)-edgesize(2) - top)/3);

slh = [0,0,0];
for ii=1:3
   cdata = zeros([imsize(2),imsize(1),3]);
   axh  = axes('Parent',figh,...
               'Units','pixels',...
               'Position',[edgesize(1),top+ii*panelheight-imsize(2),imsize],...
               'Xlim',[0.5,imsize(1)+0.5],...
               'Ylim',[0.5,imsize(2)+0.5],...
               'Visible','off');
   cdata(:,:,ii) = nrampim;
   imh  = image('Parent',axh,...
                'CData',cdata);
   axh  = axes('Parent',figh,...
               'Units','pixels',...
               'Position',[edgesize(1),top+ii*panelheight-2*imsize(2),imsize],...
               'Xlim',[0.5,imsize(1)+0.5],...
               'Ylim',[0.5,imsize(2)+0.5],...
               'Visible','off');
   cdata(:,:,ii) = rampim;
   imh  = image('Parent',axh,...
                'UserData',cdata,...
                'CData',cdata.^gamma(ii));
   txth = uicontrol('Parent',figh,...
                    'Units','pixels',...
                    'Position',[edgesize(1)+imsize(1)-ctrsz(1),top+ii*panelheight-2*imsize(2)-ctrsz(2),ctrsz],...
                    'Style','text',...
                    'HorizontalAlignment','left',...
                    'String',num2str(gamma(ii)));
   slh(ii) = uicontrol('Parent',figh,...
                       'Units','pixels',...
                       'Position',[edgesize(1),top+ii*panelheight-2*imsize(2)-ctrsz(2),imsize(1)-ctrsz(1),ctrsz(2)],...
                       'Style','slider',...
                       'Max',log(5),...
                       'Min',log(1/5),...
                       'SliderStep',[0.01,0.1],...
                       'Value',log(gamma(ii)),...
                       'UserData',struct('imh',imh,'txth',txth),...
                       'CallBack',@slider_callback);
end

% Let the user play with the GUI
waitfor(figh,'UserData');
if ~ishandle(figh)
   % The user pressed 'Cancel'
   gamma = [];
   return;
end

% Read out the settings and delete the figure
for ii=1:3
   gamma(ii) = exp(get(slh(ii),'Value'));
end
delete(figh);
disp('Setting values to dippref.');
dipsetpref('Gamma',gamma);


function slider_callback(slh,arg)
gamma = exp(get(slh,'Value'));
ud = get(slh,'UserData');
set(ud.txth,'String',num2str(gamma));
rampim = get(ud.imh,'UserData');
set(ud.imh,'CData',rampim.^gamma);
