%GET_SUBPIXEL   Retrieves subpixel values in an image
%
% USAGE:
%  out = get_subpixel(in, coord, method)
%
% PARAMETERS:
%  coord: image coordinates of interest;
%      can be a list of coordinates:
%      coord = [x' y'], x=[11.2 2.2 10.2],y=[0.1 2.3 3.7]
%  method: 'linear', 'spline', 'cubic', 'nearest'
%  out = array containing the interpolated values
%
% DEFAULTS:
%  method = 'linear'
%
% NOTE:
%  The image is converted to doubles in this function, so it
%  is not recommended for large images.

% (C) Copyright 1999-2007               Pattern Recognition Group
%     All rights reserved               Faculty of Applied Physics
%                                       Delft University of Technology
%                                       Lorentzweg 1
%                                       2628 CJ Delft
%                                       The Netherlands
%
% Bernd Rieger, Sep 2001
% Nov, 2002:     Added nD support (BR)
% 9 April 2007:  Removed use of EVAL. (CL)
% 11 April 2007: Fixed bug, INTERPN needs x and y coordinates exchanged.
% 12 April 2007: Fixed bug, test for negative coordinates was not correct.
% 13 March 2009: Fixed bug, for 1D images

function out = get_subpixel(varargin)

d = struct('menu','Point',...
           'display','Subpixel value extraction',...
           'inparams',struct('name',       {'in',         'po',       'method'},...
                             'description',{'Input image','Points',   'method'},...
                             'type',       {'image',      'image',    'option'},...
                             'dim_check',  {0,            0,           0},...
                             'range_check',{[],[],{'linear','spline','cubic','nearest'}},...
                             'required',   {1,            1,           0},...
                             'default',    {'a',          0,           'linear'}...
                             ),...
            'outparams',struct('name',{'out'},...
                               'description',{'Output image'},...
                               'type',{'image'}...
                               )...
            );
if nargin == 1
   s = varargin{1};
   if ischar(s) & strcmp(s,'DIP_GetParamList')
      out = d;
      return
   end
end
try
   [in, po, method] = getparams(d,varargin{:});
catch
   if ~isempty(paramerror)
      error(paramerror)
   else
      error(firsterr)
   end
end

po = double(po); % only way to pass >1D array is via an image :-(
po = po+1;       % shift by 1 to compensate for image->double indices, double(in)
sz = imsize(in);
nd = length(sz);

if size(po,2)~=nd & nd>1
   error('Image and point dimensions do not match.');
end
if any(any(po<1))
   error('Some coordinates are negative.');
end

if any(any(po>repmat(sz,size(po,1),1)))
   error('Coordinates out of image bounds.');
end

switch nd
   case 1
      out = interp1(double(in),po,method);
   %case 2
   %   out = interp2(double(in),po(:,1),po(:,2),method);
   %case 3
   %   out = interp3(double(in),po(:,1),po(:,2),po(:,3),method);
   otherwise
      s = cell(nd,1);
      s{1} = po(:,2); % Switching coordinates, INTERPN uses standard MATLAB y-x-z indexing.
      s{2} = po(:,1);
      for ii=3:nd
         s{ii} = po(:,ii);
      end
      out = interpn(double(in),s{:},method); % works for nd>=3
end
