%DIP_INITIALISE   Initialises the DIPimage toolbox and library
%   Add the directory this function is in to your MATLAB path,
%   then call DIP_INITIALISE.
%
%   dip_initialise('nopathenv') does not add the library path
%   to the environment variable. Do this if you set the library
%   path manually before starting MATLAB, for example if you
%   have your libraries in a non-default place. This option is
%   only effective on Windows, and will be ignored on other
%   platforms.

% (C) Copyright 1999-2009               Pattern Recognition Group
%     All rights reserved               Faculty of Applied Physics
%                                       Delft University of Technology
%                                       Lorentzweg 1
%                                       2628 CJ Delft
%                                       The Netherlands
%
% Michael van Ginkel & Cris Luengo, December 2007.
% 5 January 2008: Allowing repeated calling of this function, as before.
%                 Added MODE argument.
% 24 March 2009:  Automatically setting the library path, on Windows
%                 with MATLAB >= 7.2.

function dip_initialise(varargin)

dipbase = '';
debug = 0;
mode = '';
setpathenv = 1;
   % In order to work with MATLAB V7.4 (R2007a), MEX-files compiled on Windows (32--bit) platforms with MATLAB R11 or earlier will no longer load correctly and must be recompiled. These files can be compiled with MATLAB R12 or later.
   % To work with MATLAB V7.5 (R2007b), MEX-files compiled on any platform with MATLAB versions earlier than V7 (R14) no longer load correctly and must be rebuilt.
matlab_version_extdll_no_longer_supported=[8,0];     % Version of Windows MATLAB that does not understand MEX-files with DLL extension.
matlab_version_old_mex_no_longer_supported=[7,5];    % MATLAB 7.5 no longer supports MEX-files build with MATLAB 6.5 and earlier.
matlab_version_win_very_old_mex_not_supported=[7,4]; % MATLAB 7.4 on Windows no longer supports MEX-files build with MATLAB 5.1 and earlier.
matlab_version_highest_not_supported=[5,2];          % DIPimage does not work with MATLAB 5.2 or older.
matlab_version_has_mfilename=[6,5];                  % MFILENAME supports the 'fullpath' argument since MATLAB 6.5.
matlab_version_has_setenv=[7,2];                     % SETENV function new to MATLAB 7.2.

for ii=1:nargin
   if ~ischar(varargin{ii})
      error('Illegal input to DIP_INITIALISE: string expected');
   else
      if strcmp(varargin{ii},'debug')
         debug = 1;
      elseif strcmp(varargin{ii},'silent')
         mode = 'silent';
      elseif strcmp(varargin{ii},'nopathenv')
         setpathenv = 0;
      else
         dipbase = varargin{ii};
      end
   end
end

if exist('dip_initialise_libs')==2
   % Apparently this file already ran!
   dip_initialise_libs(mode);
   return
   % A better option might be to do DIP_EXIT, remove the dipimage and diplib paths,
   % then run this function. I'm thinking that people might add the diplib path
   % to their default PATH variable, just typing SAVEPATH will do that.
end

% Get the MATLAB version number
cv = version;
ix = find(cv=='.');
if length(ix)>1
   cv = cv(1:ix(2)-1);
elseif length(ix)<1
   cv = [cv,'.0'];
end
cv = sscanf(cv,'%d.%d');
if vnge(matlab_version_highest_not_supported,cv)
   error('MATLAB 5.3 (Release 11) or later required');
end

% Get the base directory
if isempty(dipbase)
   if vnge(cv,matlab_version_has_mfilename)
      dipbase = mfilename('fullpath');
   else
      dipbase = which(mfilename);
   end
   dipbase = fileparts(fileparts(dipbase)); % remove M-file name and go up a directory.
else
   % test to see if the user is being truthful
   while dipbase(end)==filesep
      dipbase(end) = [];
   end
   tf = fullfile(dipbase,'dipimage','dip_initialise.m');
   if ~exist(tf,'file')
      error('Illegal input to DIP_INITIALISE: path does not point to the DIPimage base directory.')
   end
end

% Find the library search path
if ~vnge(cv,matlab_version_has_setenv)
   setpathenv = 0;
end
if isunix
   setpathenv = 0;
end
if setpathenv
   libpath = fileparts(dipbase);
   switch computer
      case 'PCWIN'
         libpath = fullfile(libpath,'win32','lib');
      case 'PCWIN64'
         libpath = fullfile(libpath,'win64','lib');
      otherwise
         % This should not happen.
         error('Unsupported platform!')
   end
   envvar = 'PATH';
   envsep = ';';
end

% Find the directory compiled with the latest version of MATLAB that
% is equal or older than this version.
vds = dir(fullfile(dipbase,'mlv*'));
vds = {vds([vds.isdir]).name};
exts={mexext};
if strcmp(mexext,'mexw32') & ~vnge(cv,matlab_version_extdll_no_longer_supported)
   exts{2}='dll';
end
for ii=1:length(vds)
   candidate = [-1,0]; % not a good directory!
   % Test to see if the version directory contains MEX-files for this platform & version.
   for mi=1:length(exts)
      if exist(fullfile(dipbase,vds{ii},'dipimage_mex',['di_forcebin.',exts{mi}]),'file')
         candidate = sscanf(vds{ii},'mlv%d_%d');
         break;
      end
   end
   % If this is MATLAB version >= 7.5, then discard any directories that are < 7.0.
   if vnge(cv,matlab_version_old_mex_no_longer_supported) & ~vnge(candidate,[7,0])
      %candidate = [-1,0]; % BOLLOCKS! It works for me!!!
   end
   % If this is MATLAB version >= 7.4 on Windows, then discard any directories that are < 6.0.
   if ~isunix & vnge(cv,matlab_version_win_very_old_mex_not_supported) & ~vnge(candidate,[6,0])
      candidate = [-1,0];
   end
   vds{ii}=candidate;
end
lv = [0;0]; % initialise to unusable number
for ii=1:length(vds)
   if vnge(vds{ii},lv) & vnge(cv,vds{ii})
      lv = vds{ii};
   end
end
if all(lv==0)
   error('DIPimage does not currently support your version of MATLAB on your platform. Please see http://www.diplib.org/')
end
vdir = sprintf('mlv%d_%d',lv);

% Add paths and initialise DIPlib libraries
if setpathenv
   setenv(envvar,[libpath,envsep,getenv(envvar)]);
end
addpath(fullfile(dipbase,vdir,'dipimage_mex'));
if debug
   addpath(fullfile(dipbase,vdir,'diplib_dbg'));
else
   addpath(fullfile(dipbase,vdir,'diplib'));
end
dip_initialise_libs(mode);
dipsetpref('CommandFilePath',fullfile(dipbase,vdir,'dipimage_mex'));


function r = vnge(a,b) % >=
r = 0;
if ( a(1)>b(1) ) | ( a(1)==b(1) & a(2)>=b(2) )
   r = 1;
end
